{*************************************************************
**************************************************************
*   Componente TBuscador   (Modulo principal del componente) *
*   Objetivo: Busqueda de archivos con multiseleccin de     *
*             rutas. Ejemplo sobre el uso de Hilos           *
*             Captulo V de la Serie Threads                 *
*                                                            *
*   Autor: Salvador Jover      mailto: s.jover@wanadoo.es    *
*          Jose Manuel Navarro mailto:jmnavarro@lexnova.es   *
*                                                            *
*   Revista Sintesis N 15  http://www.GrupoAlbor.com/       *
**************************************************************
**************************************************************
* Este componente debe ser tomado como un ejemplo, y no esta *
* por ello libre de errores; por lo que, como autores, no    *
* no podemos garantizar ni recomendar su uso fuera de lo que *
* establece el aprendizaje.                                  *
* As pues, deben establecerse siempre las suficientes       *
* reservas y cautelas a tal efecto. Su uso es libre.         *
**************************************************************}
// VER ULTIMA NOTA EN ARCHIVO "LEEME.TXT"
{ OBJETIVOS DEL COMPONENTE:
   Bsqueda de archivos mediante mltiples hilos: nico hilo por
   ruta, con creacin de hilo para cada ruta.
   Permite la busqueda en unidades de red conectadas a la nuestra.
   Admite el uso de comodines '*' y '?'}

{NOTA DE ATENCION SOBRE SU USO:
  En esta fase de diseo del componente, no debe ser usado aceptando
  como padre (Parent) un componente que no sea un TForm. De no hacerlo
  as, se debera activar el evento OnCloseQuery del formulario y antes
  de que sea cerrado, proceder a Cancelar la ejecucin de la busqueda
  mediante el mtodo Cancel, para as evitar una posible excepcin.

  Debe ser incluido un solo buscador por Form}


unit buscador;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, comctrls, stdctrls, imglist;

const

  // Mensajes de comunicacin entre TBuscador e THiloBusqueda
  // -La comunicacin se produce siempre desde sentido del hilo al buscador-
  //
  BA_INCPOS  = WM_USER + 1000;  // Incrementa valor POSICION de Barra
  BA_INCMAX  = WM_USER + 1001;  // Incrementa valor MAXIMO de Barra

  RE_NUMITEM = WM_USER + 1002;  //  Aade un nuevo TListItem a los resultados
  RE_SELECT  = WM_USER + 1003;  //  Selecciona TListItem de los resultados
  RE_ADD     = WM_USER + 1004;  //  Aade un nuevo hilo a la lista de Hilos
  RE_DEL     = WM_USER + 1005;  //  Elimina un hilo de la lista de hilos

  AR_ADD     = WM_USER + 1006;  //  Aade un objeto TTreeNode al Arbol

  BU_ABORT   = WM_USER + 1007;  //  Se ha producido un error o cancelacion
  BU_END     = WM_USER + 1009;  //  Busqueda ha acabado sin incidencias

type

  //Crear un descendiente del Objeto TListView nos ayudar a acceder
  //a mtodos protegidos
  TListaVista = class(TListView)
  end;

  {Declaramos dos tipos de Excepcion}
  ERutaVacia  = class(Exception);
  ETokenVacio = class(Exception);

  {El algoritmo de bsqueda puede finalizar de forma no natural por
   cualquiera de estas tres razones. Pueden existir ms. Nosotros
   solo vamos a considerar que la ruta entregada sea vacia, que el
   fichero a buscar sea una cadena vacia o bien que el usuario
   decida cancelarlo de forma anticipada.}
  TIncidencia = (inNoIncidencia, inRutaVacia, inTokenVacio, inCancelacion, inDuplicado);

  {El estado nos permite discriminar cuando est el proceso de hilos
  ejecutandose. Se pone en estado de ejecucin nada ms iniciarse el
  mtodo DoExecute() y cualquier incidencia o fin de la busqueda lo
  pone de nuevo en estado inactivo.}
  TEstado = (esInactivo, esEjecucion);

  {Definimos un nuevo tipo de notificacin en la que se le comunicar
   al usuario el error que se ha producio en la entrega de los parmetros
   segn los requermientos que hemos establecido.}
  TAbortEvent = procedure (Sender: TObject; AIncidencia: TIncidencia) of Object;

  //Se aadirn posiblemente ms adelante. Estos eventos se produciran
  //cada vez que un nuevo nodo es aadido y cada vez que resuelve una
  //coincidencia.
  TFindAItem = procedure (Sender: TObject; const AItem: String) of Object;
  TSearchAItem = procedure (Sender: TObject; const AItem: String) of Object;

  //Anticipamos la declaracin de clase
  TBuscador = class;

  {Este registro ser guardado dentro de una lista de punteros, almacenando
   los parmetros de cada una de las busquedas a efectuar. Cuando se crea
   un objeto TBusqueda, tiene dos misiones diferenciadas que son:
   1- Aade a la lista de punteros las rutas de todas las carpetas que
       contiene
   2- Busca en sus ficheros si existe alguna coincidencia con la cadena
       a buscar.
   Antes de finalizar, si la lista contiene todava elementos [count > 0],
   el objeto TBusqueda deber crear otro objeto TBusqueda que recibe como
   parmetros los datos del primer item de la lista. A continuacin proceder
   a eliminar dicho nodo de la lista, garantizando as la busqueda a travs
   de todos los directorios implicados.
   Esto se producir siempre y cuando se haya incluido la opcin de Explorar
   las Subcarpetas.}

  PNodo = ^TNodo;
  TNodo = Record
     NBuscador: TBuscador;
     NNodo: TTreeNode;
     NArgumentos: array[0..MAX_PATH-1] of char;
     NDesciende: Boolean;
  End;

  TBuscador = class(TWinControl)

  private

    fArbol:        TTreeView;     //rbol visual de la busqueda
    fBarra:        TProgressBar;  //barra de progreso visual
    fResultados:   TListaVista;   //lista que visualiza resultados

    // Lista de cadenas de rutas sobre las que se efectuar las bsquedas
    // Su uso es anecdtico y se ha hecho as por falta de tiempo en
    // el diseo del componente. La lista de rutas podria ser almacenada
    // en una lista TStrings, pudiendo ser editada visualmente en tiempo
    // de diseo, como lo podemos hacer mediante un TMemo por ejemplo
    // en su propiedad Lines.
    // Tambin se le podra incluir en el editor de propiedades un
    // selector de carpetas (Boton de Puntos suspensivos [...]) similar
    // al que se ha incluido al pulsar el botn aadir del form del ejemplo.
    // En Delphi 6 ya se incluyen en la pestaa samples un selector que
    // adems podra hacer multiseleccin de archivos.
    // Si deseis emplearlo tan solo tenis que crear un descendiente y
    // redefinir la propiedad multiselec como Publicada o Publica para hacer
    // uso de ella.
    // Intentaremos verlo ms adelante si es posible, como podemos crear
    // o definir dicho editor de propiedades
    fListaRutas:  TStringList;

    fToken:       string;     //archivo a buscar
    fSubcarpetas: boolean;    //hay que descender en la bsqueda?

    fMSecs:       LongWord;   // datos estadsticos

//Listas de uso interno del componente
    fListaThreads:     TList;      // lista interna de hilos

//Eventos:
    fOnEnd: TNotifyEvent;     // al finalizar la busqueda
    fOnAbort: TAbortEvent;    // al cancelarse la ejecucin de la busqueda

// Variable que almacena el estado del componente
    fEstado: TEstado;

// Almacenan los valores de las propiedades respectivas
    fArbolVisible: Boolean;   // est visible el arbol?
    fImage: TCustomImageList; // imagen aadida al arbol [icono]
    fColor: TColor;           // color Background

    {Nota sobre el color:
    Inici tan solo la primera linea de prueba, activando el color
    de background pero habra que hacerlo bien. Debera estar disponible
    el atributo Font para seleccionar el tamao de la letra y el color
    por ejemplo. Habra que valorar que atributos nos hara falta en el
    nuevo componente: necesitaramos tambien eventos que no he definido
    por falta de tiempo. Por ejemplo el evento OnClick de cada componente
    podra ser reconducido a tres OnClick nuevos (OnClickBarra, OnClickArbol,
    OnClickResultados) y as sucesivamente con aquellos eventos que nos
    gustara que fueran pblicos y que quedan ocultos por TWinControl.}

    // Los procedimientos de escritura de las propiedades podemos ocultarlos
    // en la zona privada
    procedure SetToken(const Value: String);
    procedure SetArbolVisible( Value: Boolean);
    procedure SetImage(const Value: TCustomImageList);
    procedure SetColor(const Value: TColor);
    // Los mensajes tambin
    procedure baIncPos( var Msg: TMessage);  message BA_INCPOS;
    procedure baIncMax( var Msg: TMessage);  message BA_INCMAX;
    procedure reNumItem( var Msg: TMessage); message RE_NUMITEM;
    procedure reSelect( var Msg: TMessage);  message RE_SELECT;
    procedure reAdd( var Msg: TMessage);     message RE_ADD;
    procedure reDel( var Msg: TMessage);     message RE_DEL;
    procedure arAdd( var Msg: TMessage);     message AR_ADD;

  protected

    // Este manejador de mensaje ya es protegido. No podemos ocultar visibilidad
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;

    // Declaramos virtuales aquellos mtodos que pueden ser objeto
    // de nuevas revisiones en los descendientes
    procedure DoExecute; virtual;
    procedure DoCancel; virtual;
    procedure AddThread(Hilo: TThread); virtual;
    procedure DeleteThread(Hilo: TThread); virtual;

    // Lectura de la barra de progreso
    property BarraProgreso  :  TProgressBar read fBarra;

  public

    // interfaz pblica del componente
    // constructor y destructor sobrescritos
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure Add(const NewPath: string);   // aadir ruta
    procedure Execute;                // iniciar ejecucion
    procedure Pause(parar: boolean);  // pausar la bsqueda
    procedure Cancel;                 // cancelar bsqueda

    property ArbolResultados:  TTreeView    read FArbol;
    property MSecs          :  LongWord     read FMSecs    default 0;
    property Estado         :  TEstado      read FEstado   default esInactivo;

  published

   //Se ha finalizado la busqueda en un directorio
    property OnEnd: TNotifyEvent      read FOnEnd        write FOnEnd;
    //Se ha producido una incidencia
    property OnAbort: TAbortEvent     read FOnAbort      write FOnAbort;

    property Align;
    property Color: TColor            read FColor        write SetColor;
    property Image: TCustomImageList  read FImage        write SetImage;
    property ArbolVisible: Boolean    read fArbolVisible write SetArbolVisible;
    property Token: String            read FToken        write SetToken;
    property HaySubcarpetas: Boolean  read FSubcarpetas  write FSubcarpetas default True;
  end;

   {Funcion Gancho: Capturar un gancho de tipo WH_CALLWNDPROC
      nCode: Si > 0 entregar a sig. gancho. En caso contrario
             procesar nosotros el mensaje
      wParam: 0 = Mensaje viene del proceso actual
              <> 0 = de otro proceso
      lParam: ^TCWPStruct = packed Record

    Esta funcin gancho siempre devuelve 0                     }

    function WHCALLWNDPROC(nCode: Integer; wParam: WPARAM; lParam: LPARAM): LRESULT; stdcall;

var
  FListaNodos:  TList;   // almacena las lista de rutas pendientes
                         // de explorar
  WinHook: HHOOK;        // GANCHO

  {Nota aclarativa sobre el uso del Hook (gancho)
   Un problema que me ha resultado de muy difcil anlisis y solucin no
   ha sido tanto la sincronizacin de los hilos sino la respuesta de estos
   ante el evento de destruccion del formulario. Al cerrarse la ventana,
   se producan excepciones de aquellos hilos en ejecucin con acceso a
   objetos del form (barra/arbol y lista de resultados). El motivo, segn
   a mi humilde entender, viene propiciado por la destruccin de la ventana,
   por lo que entend que capturando el evento OnClose del formulario
   mediante un Gancho, este problema desaparecera. Segn entiendo esto es
   as y funciona bien pero tiene un problema: SI SITUAIS EL COMPONENTE
   DENTRO DE OTRO OBJETO (COMO UN PANEL POR EJEMPLO) FALLAR Y SE PRODUCIR
   UNA EXCEPCION AL CERRAR LA VENTANA. As que por favor, tenerlo en cuenta.
   Una posible solucin a esto podra ser impedir que fuera "soltado" sobre
   otro componente del form a no ser que fuera el mismo form.
   Intentaremos tambin ms adelante modificar esto para que tenga un
   comportamiento correcto. }

procedure Register;

implementation

uses HiloBusqueda;

// Procedimiento de registro del componente
Procedure Register;
begin
   RegisterComponents('GrupoAlbor', [TBuscador]);
end;

var
varBuscador: TBuscador;   //representa a si mismo

{TBuscador}

//--------------------------------------------------------------------
// Nombre    : function WHCALLWNDPROC
// Objetivo  : Funcin Hook. Capturamos el mensaje WM_CLOSE, ligado al
//             cierre del formulario
// Comentario: Hemos propuesto esta alternativa ante el problema que se genera
//             al ser destruido el formulario y dejar de ser valido el acceso
//             al mismo. En ese momento, algunos hilos, todava desconocen esta
//             situacin y generarn excepciones la intentar acceder al arbol,
//             a la lista de resultados o bien a la barra de progreso.
//             Debe ser tomada como una propuesta. Intentaremos profundizar a lo
//             largo del desarrollo de este ejemplo en esta cuestin, que nos
//             llevar  varios artculos.
//
function WHCALLWNDPROC(nCode: Integer; wParam: WPARAM; lParam: LPARAM): LRESULT; stdcall;
var
   Estructura: ^TCWPStruct;
begin
   if nCode > -1 then begin   //Si nos corresponde procesar la informacin
      if wParam = 0 then begin  // Si el mensaje proviene de este proceso
         Estructura:= Pointer(lParam);  //se cierra la ventana?
         if Estructura.message = WM_CLOSE then varBuscador.DoCancel;
      end;
      Result:= 0; // No procesamos el mensaje  (valor siempre 0)
   end  //si no nos corresponde, que otro gancho lo intente
   else Result:= CallNextHookEx(WinHook, nCode, wParam, lParam);
end;


//--------------------------------------------------------------------
// Nombre    : constructor Create
// Objetivo  : Creacin del componente.
// Comentario:
//
constructor TBuscador.Create(AOwner: TComponent);
var
   ListColumn: TListColumn;
begin
   inherited Create(AOwner);    // invocacin del constructor heredado

   varBuscador:= Self; // apunta a si mismo. Nos servir para ejecutar Cancel
                       // antes de que sea cerrada la ventana.
                       // Ojo: No puede existir mas de un buscador en el Form
                       // Consultarlo en el artculo
                       //*****************************************************

   fResultados:= TListaVista.Create(Self);  // LISTA RESULTADOS
   fResultados.Parent:= Self;

   fResultados.ViewStyle:= vsReport;
   ListColumn:= fResultados.Columns.Add;
   ListColumn.Caption:= 'Nombre';
   ListColumn.AutoSize:= True;
   fResultados.Align:= alClient;
   fResultados.Update;

   fBarra:= TProgressBar.Create(Self);      // BARRA PROGRESO
   fBarra.Parent:= Self;

   fBarra.Smooth:= True; //no segmentada
   fBarra.Max:= 0;
   fBarra.Position:= 0;
   fBarra.Align:= alBottom;

   fArbol:= TTreeView.Create(Self);         // ARBOL
   fArbol.Parent:= Self;

   fArbol.Align:= alRight;

       //la que contiene los nodos de busquedas pendientes
   fListaNodos:= TList.Create;
       //la que contiene las cadenas de busquedas pendientes
   fListaRutas:= TStringList.Create;
       //la que contiene las busquedas en ejecucion
   fListaThreads := TList.Create;

   fEstado:= esInactivo; // estado inactivo

   fColor:= clWindow;    // color clWindow
   fArbolVisible:= True; // arbol visible
   fSubcarpetas:= True;  // queremos que se busque en subcarpetas
   // finalmente activamos el gancho
   WinHook:= SetWindowsHookEx(WH_CALLWNDPROC, @WHCALLWNDPROC, 0, GetCurrentThreadId);
end;


//--------------------------------------------------------------------
// Nombre    : destructor Destroy;
// Objetivo  : Destructor del componente.
// Comentario: Es liberada la memoria reservada dinmicamente, as como aquellos
//             objetos creados de igual forma.
//
destructor TBuscador.Destroy;
var
   i: Integer;
   tmp_Nodo: PNodo;
begin
   //Descargamos el gancho solo en el caso de que se haya instalado
   if WinHook <> 0 then UnHookWindowsHookEx(WinHook);

   fEstado:= esInactivo;

//destruccin de los objetos creados dinmicamente

   //de la lista de cadenas de rutas pendientes
   if Assigned(fListaRutas) then begin
      fListaRutas.Clear;
      fListaRutas.Free;
   end;

   fImage:= nil;    // desasignamos la imagen asociada

   //de los nodos y de la lista de nodos
   for i := FListaNodos.Count-1 downto 0 do begin
      tmp_Nodo:= PNodo(FListaNodos.Items[i]);
      Dispose(tmp_Nodo);
      fListaNodos.Items[i]:= Nil;
   end;
   fListaNodos.Clear;
   fListaNodos.Free;
   fListaNodos:= Nil;

   //de los busquedas en ejecucion (hilos), eliminando la lista
   if fListaThreads <> nil then begin
      for i := fListaThreads.Count - 1 downto 0 do begin
         TThread(fListaThreads[i]).Terminate;
         fListaThreads[i]:= Nil;
      end;
      fListaThreads.Clear;
   end;
   fListaThreads.Free;
   fListaThreads:= Nil;

   // destruimos los objetos y componentes creados dinmicamente
   fBarra.Free;
   fArbol.Free;
   fResultados.Free;
   inherited Destroy;  // invocacion del destructor en el ascendente
end;


//--------------------------------------------------------------------
// Nombre    : procedure Execute
// Objetivo  : Ejecucin del algoritmo de bsqueda
// Comentario: Envuelve una invocacin del mtodo DoExecute declarado
//             como virtual.
//
procedure TBuscador.Execute;
begin
      DoExecute;
end;


//--------------------------------------------------------------------
// Nombre    : procedure DoExecute
// Objetivo  : Ejecucin del algoritmo de bsqueda
// Comentario: Virtual.
//             Al ser declarado como virtual y protegido puede ser redefinido
//             por un descendiente para hacer uso de otras acciones necesarias
//
procedure TBuscador.DoExecute;
var
   varBusqueda: TBusqueda;
   s: String;
begin
   fMSecs := 0;   // ponemos a cero el contador de tiempo
   fMSecs := GetTickCount;
   fEstado:= esEjecucion; // iniciamos el estado de ejecucin
   fArbol.Items.Clear;    // vaciamos el arbol visual
   fResultados.Items.Clear; // vaciamos la lista de resultados
   // PRECONDICION DEL ALGORITMO
   // Abortamos lanzamiento. Ya est en ejecucin
   if (fListaThreads.Count > 0) then begin
      DoCancel;
      if Assigned(FOnAbort) then FOnAbort(Self, inDuplicado);
      Raise Exception.Create('Error: Busqueda en ejecucion');
   end;
   // Abortamos la ejecucin. No existe token.
   if (FToken = '') or (FToken = ' ') then begin
      DoCancel;
      if Assigned(FOnAbort) then FOnAbort(Self, inTokenVacio);
      Raise ERutaVacia.Create('Error: Token vaco');
   end;
   // Abortamos la ejecucin. La ruta est vacia.
   if FListaRutas.Count = 0 then begin
      DoCancel;
      if Assigned(FOnAbort) then FOnAbort(Self, inRutaVacia);
      Raise ERutaVacia.Create('Error: Ruta vaca');
   end;
   // proceso de creacin de los objetos Hilos (THiloBusqueda)
   // para cada una de las rutas elegidas por el usuario
   While FListaRutas.Count > 0 do begin
      s:= fListaRutas[0];  // almacenamos en [s] la primera ruta
      fListaRutas.Delete(0); // ya podemos borrarla y se resituaran de nuevo
      fBarra.Max:= FBarra.Max + 1;  //aadimos al valor mximo
      fBarra.Position:= fBarra.Position + 1;  // y la posicin de la barra
      if s[length(s)] <> '\' then s:= s + '\';  // nos aseguramos que es correcto el final en '\'
      // creacin del hilo
      varBusqueda := TBusqueda.Create(Self, nil, s + FToken, FSubcarpetas);
      // y lo aadimos a la lista de hilos
      fListaThreads.Add(varBusqueda);
      varBusqueda.Resume; // y ya podemos lanzar su ejecucin
   end;  // el proceso se repite mientras queden rutas

end;


//--------------------------------------------------------------------
// Nombre    : procedure Add
// Objetivo  : Aadir una ruta al componente Buscador
// Comentario: Evalua que no sea una cadena vaca o nula [PRECONDICION]
//
procedure TBuscador.Add(const NewPath: string);
begin
   if (NewPath = '') or (NewPath = ' ') then begin
      if Assigned(FOnAbort) then FOnAbort(Self, inRutaVacia);
      Raise ERutaVacia.Create('Error: Ruta Vaca. Introduce un valor correcto');
   end;
   // solo si la consideramo correcta
   // (solo pedimos en este caso que sea no vacia o no nula)
   fListaRutas.Add(NewPath);
end;


//--------------------------------------------------------------------
// Nombre    : procedure SetToken
// Objetivo  : Mtodo de escritura de la propiedad Token que almacena el fichero
//             a buscar
// Comentario: Evalua que no sea una cadena vaca o nula [PRECONDICION]
//
procedure TBuscador.SetToken(const Value: String);
begin
   if (Value = '') or (Value = ' ') then begin
       if Assigned(FOnAbort) then FOnAbort(Self, inTokenVacio);
       Raise ERutaVacia.Create('Error: Token Vaco. Introduce un valor correcto');
   end;
   // solo si la consideramo correcta
   fToken := Value;
end;


//--------------------------------------------------------------------
// Nombre    : procedure AddThread
// Objetivo  : Aadir un nuevo TThread a la lista de Hilos
// Comentario: Virtual
//             Es parte de la condicin de finalizacin del algoritmo
//             El algoritmo finaliza correctamente cuando la lista de hilos
//             y la lista de nodos de rutas pendientes, estn vacias.
//
procedure TBuscador.AddThread( Hilo: TThread);
begin
   //aadimos el nuevo hilo
   if fListaThreads.IndexOf(Hilo) = -1 then fListaThreads.Add(Hilo);
end;


//--------------------------------------------------------------------
// Nombre    : procedure DeleteThread
// Objetivo  : Suprimir un hilo de la lista de Hilos
// Comentario: Virtual
//             Antes de ser destruido el hilo, se busca a si mismo en la lista
//             de hilos para eliminarse de ella.
//
procedure TBuscador.DeleteThread(Hilo: TThread);
var
   SinHilos: boolean;
   ind:      integer;
begin
   ind := fListaThreads.IndexOf(Hilo); // indice de la bsqueda
   if ind <> -1 then fListaThreads.Delete(ind);  // me elimino de la lista

   // evaluo si quedan hilos para poder lanzar el evento de finalizacin
   // implementado por el usuario del componente (OnEnd)
   // CONDICIONES DE FINALIZACION DEL ALGORITMO
   SinHilos := (fListaThreads.Count = 0) and (fListaNodos.Count = 0);
   // si ya no quedan hilos, lanzo el evento en cuestin
   if SinHilos then begin
   //En este punto debera haber acabado el algoritmo de bsqueda
   //La lista de nodos y la lista de hilos deben de estar vacas
      fEstado:= esInactivo; //inicializamos los estados
      fBarra.Max:= 0;
      fBarra.Position:= 0;
      fMSecs := GetTickCount - fMSecs;
      fArbol.Items.BeginUpdate;
      fArbol.FullExpand;              // desplegamos el arbol
      fArbol.Items.EndUpdate;
      if Assigned(FOnEnd) then FOnEnd(Self); // lanzamos el evento OnEnd
      //que le da al usuario de inicializar la interfaz del formulario ante
      // la finalizacin del algoritmo
      // CUADRO EMERGENTE DE TOMA DE TIEMPOS  -ACCESORIO-
      MessageBox(GetActiveWindow, PChar(FormatFloat('#,0', fResultados.Items.Count) + ' coincidencias en un total de '+
                                        FormatFloat('#,0', fArbol.Items.Count) + ' carpetas.' + #13 + 'Tiempo empleado: ' +
                                        FormatFloat('#,0', fMSecs) + ' milisegundos.'),
                                        'Atencin', MB_ICONINFORMATION);
      // Este cuadro de toma de tiempos desaparecer en la versin final del
      // componente y tan solo est activado mientras el mismo est en fase de
      // diseo y pruebas
   end;
end;


//--------------------------------------------------------------------
// Nombre    : procedure Pause
// Objetivo  : Pausa o Activacin de la ejecucin de la bsqueda
// Comentario:
//
procedure TBuscador.Pause(parar: boolean);
var
   i: integer;
begin
   if parar then
      for i := fListaThreads.Count - 1 downto 0 do
         TThread(fListaThreads[i]).Suspend
   else
      for i := fListaThreads.Count - 1 downto 0 do
         TThread(fListaThreads[i]).Resume;
end;


//--------------------------------------------------------------------
// Nombre    : procedure Cancel
// Objetivo  : Mtodo para la cancelacin de la ejecucin de la bsqueda
// Comentario: Envuelve una invocacin del mtodo DoCancel declarado
//             como virtual.
//
procedure TBuscador.Cancel;
begin
   if fEstado = esInactivo then exit;
   DoCancel;
   if Assigned(FOnAbort) then FOnAbort(Self, inCancelacion);
end;


//--------------------------------------------------------------------
// Nombre    : procedure DoCancel
// Objetivo  : Cancelar la ejecucin del algoritmo de bsqueda
// Comentario: Virtual
//
procedure TBuscador.DoCancel;
var
   i: Integer;
   tmp_Nodo: PNodo;
begin
   fEstado:= esInactivo; //inicializamos el estado
   // inicializamos la barra de progreso
   fBarra.Max:= 0;
   fBarra.Position:= 0;
   // inicializamos la lista de nodos. Deben ser eliminados todos los nodos
   for i := FListaNodos.Count-1 downto 0 do begin
      tmp_Nodo:= PNodo(FListaNodos.Items[i]);
      Dispose(tmp_Nodo);
      FListaNodos.Items[i]:= Nil;
   end;
   fListaNodos.Clear;
   // inicializamos la lista de Hilos. Deben terminar
   for i := fListaThreads.Count - 1 downto 0 do begin
      TThread(fListaThreads[i]).Terminate;
      fListaThreads[i]:= Nil;
   end;
   fListaThreads.Clear;
   // inicializamos la lista de rutas
   if Assigned(FListaRutas) then begin
      FListaRutas.Clear;
   end;
end;


//--------------------------------------------------------------------
// Nombre    : procedure SetArbolVisible
// Objetivo  : Metodo de escritura de la propiedad ArbolVisible
// Comentario:
//
procedure TBuscador.SetArbolVisible(Value: Boolean);
begin
   if fArbolVisible <> Value then fArbolVisible:= Value;
   if fArbolVisible then
      fArbol.Width:= Width div 2
   else
      fArbol.Width:= 0;
end;


//--------------------------------------------------------------------
// Nombre    : procedure WMSize
// Objetivo  : Respuesta del componente ante un cambio en sus dimensiones
// Comentario: Este mtodo se modificar en versiones posteriores para
//             darle mayor flexibilidad.
//             El usuario debera ajustar con el raton en tiempo de diseo
//             y de ejecucin la proporcin de lista de resultados visible.
//             La idea es que sea su comportamiente similar al componente
//             TSplitter
//
procedure TBuscador.WMSize(var Message: TWMSize);
begin
   inherited;
   if fArbolVisible then begin
      fResultados.Height:= Height - fBarra.Height;
      fResultados.Width:= Width div 2;
      fBarra.Top:= Height - fBarra.Height;
      fBarra.Width:= Width;
      fArbol.Left:= FResultados.Width;
      fArbol.Height:= FResultados.Height;
      fArbol.Width:= Width div 2;
   end
   else begin
      fResultados.Height:= Height - fBarra.Height;
      fResultados.Width:= Width;
      fBarra.Top:= Height - fBarra.Height;
      fBarra.Width:= Width;
      fArbol.Left:= FResultados.Width;
      fArbol.Height:= FResultados.Height;
   end;
end;


//--------------------------------------------------------------------
// Nombre    : procedure SetColor
// Objetivo  : Mtodo de escritura de la propiedad Color
// Comentario: Hemos necesitado extender el componente TListView para poder
//             tener acceso al mtodo protegido UpdateColumns que actualiza
//             los componentes internos al TListView que deben ser repintados.
//             Suprimir el mtodo y valorad el efecto.
//
procedure TBuscador.SetColor(const Value: TColor);
begin
   fColor:= Value;
   fArbol.Color:= Value;
   fResultados.Color:= Value;
   fResultados.UpdateColumns; //  <--- Accedemos a un mtodo protegido
end;


//--------------------------------------------------------------------
// Nombre    : procedure SetImage
// Objetivo  : Mtodo de escritura de la propiedad Image
// Comentario: Nos permite asignar una imagen al arbol visual.
//
procedure TBuscador.SetImage(const Value: TCustomImageList);
begin
   fImage := Value;
   fArbol.Images:= Value;
end;


//--------------------------------------------------------------------
// Nombre    : procedure baIncPos
// Objetivo  : Incrementar la posicin de la barra de progreso
// Comentario: Se ha explorado una nueva carpeta
//             otra interpretacion: Finaliza un hilo
//
procedure TBuscador.baIncPos(var Msg: TMessage);
begin
   fBarra.Position:= fBarra.Position + 1;
end;


//--------------------------------------------------------------------
// Nombre    : procedure baIncMax
// Objetivo  : Incrementar la propiedad Maxima de la barra de progreso
// Comentario: Se ha aadido un nuevo nodo a la lista de nodos
//             otra interpretacin: Un nuevo hilo que crear
//             El matiz para comprender esto, es que cada hilo representa
//             una carpeta que debe ser analizada y el total de hilos
//             creados durante el algoritmo debe ser igual al nmero
//             de carpetas exploradas
//
procedure TBuscador.baIncMax(var Msg: TMessage);
begin
   fBarra.Max:= fBarra.Max + 1;
end;


//--------------------------------------------------------------------
// Nombre    : procedure reNumItem
// Objetivo  : Obtener un nuevo TlistItem de la lista de Resultados.
// Comentario: Le pedimos al componente buscador que nos aada un TListItem
//             a la lista de resultados para que el hilo de bsqueda lo manipule
//             entregandole la ruta del fichero encontrado
//
procedure TBuscador.reNumItem(var Msg: TMessage);
begin
   Msg.Result:= Integer(fResultados.Items.Add);
end;


//--------------------------------------------------------------------
// Nombre    : procedure reSelect
// Objetivo  : Seleccionar un resultado de la Lista de Resultados
// Comentario:
//
procedure TBuscador.reSelect(var Msg: TMessage);
begin
   fResultados.Selected:= TListItem(Msg.WParam);
end;


//--------------------------------------------------------------------
// Nombre    : procedure reAdd
// Objetivo  : Aadir un nuevo hilo a la lista de hilos
// Comentario: Es un envolvente del metodo AddThread() dado que
//             este no puede ser accedido por ser protegido
//
procedure TBuscador.reAdd(var Msg: TMessage);
begin
   AddThread(TThread(Msg.WParam));
end;


//--------------------------------------------------------------------
// Nombre    : procedure reDel
// Objetivo  : Eliminar un hilo de la lista de hilos
// Comentario: Es un envolvente del metodo AddThread() dado que
//             este no puede ser accedido por ser protegido
//
procedure TBuscador.reDel(var Msg: TMessage);
begin
   DeleteThread(TThread(Msg.WParam));
end;


//--------------------------------------------------------------------
// Nombre    : procedure arAdd
// Objetivo  : Informar al usuario de la situacin de busqueda en el rbol
// Comentario: Accedemos a los objetos del componente Buscador mediante
//             el envio de un mensaje singular para ambos.
//
procedure TBuscador.arAdd(var Msg: TMessage);
var
   s: String;
   MiNodo: TTreeNode;
begin
   s:=  String(Msg.LParam);
   If Msg.WParam = 0 then
      MiNodo:= FArbol.Items.Add( Nil, 'Buscando en "' + s + '"...')
   else begin
      MiNodo:= FArbol.Items.AddChild(TTreeNode(Msg.WParam), 'Buscando en "' + s + '"...');
   end;
   Msg.Result:= Integer(MiNodo);
end;


end.
